﻿#if       !defined(INCLUDED_SZ_STORED_ITEM)
#define            INCLUDED_SZ_STORED_ITEM

#if       !defined(INCLUDED_SZ_COMMON)
#include  <szCommon.hpp>
#endif // !defined(INCLUDED_SZ_COMMON)

#if       !defined(INCLUDED_SZ_FILE_ATTRIBUTES)
#include  <szFileAttributes.hpp>
#endif // !defined(INCLUDED_SZ_FILE_ATTRIBUTES)

#if       !defined(INCLUDED_SZ_TIME)
#include  <szTime.hpp>
#endif // !defined(INCLUDED_SZ_TIME)

SZ_NS_BEG(szpp)

class SZ_SPEC StoredItemContainer;

/// <summary>
/// <para>
/// アーカイブ内の一つの格納アイテムを表現するクラス。
/// </para>
/// <para>
/// 一般には「格納アイテム＝格納ファイル」と考えて良いが、tar 系のアーカイブの場合は
/// ディレクトリやシンボリックリンクなどの格納アイテムも出現する。
/// </para>
/// </summary>
/// <remarks>
/// 本クラスはあくまで単一の格納アイテムの情報を保持するためのものであり、
/// 格納アイテムが他の格納アイテムを含むような階層構造を表現することは想定していない。
/// そのような目的には StoredItemContainer クラスによる階層構造表現を利用すればよい。
/// </remarks>
/// <notes>
/// リンクパスはほとんどのアイテムには不要なフィールドなので、メンバとしては保持しない。
/// また、ファイルコメントもほとんどのアイテムに不要なので、メンバとしては保持しない。
/// </notes>
class SZ_SPEC StoredItem : boost::noncopyable
{
public:

  /// デフォルトコンストラクタ。
  StoredItem();
  /// インデックスを指定するコンストラクタ。
  StoredItem(index_t index);

  /// <summary>
  /// （一般には相対パス表現されている）格納アイテムのファイル／ディレクトリ名文字列を取得するメソッド。
  /// </summary>
  /// <remarks>
  /// ディレクトリアイテムであったとしても末尾にバックスラッシュは<b>付けない</b>。
  /// ディレクトリかどうかはあくまでファイル属性で判断する。
  /// </remarks>
  const szstring &GetName() const { return name; }
  /// <summary>
  /// 格納アイテムのファイル／ディレクトリ名文字列を設定するメソッド。通常、クライアントが本メソッドを呼ぶ必要はない。
  /// </summary>
  void            SetName(const szstring &name);

  /// 格納アイテムの実際のサイズを取得するメソッド。
  u64  GetActualSize() const { return actualSize; }
  /// 格納アイテムの実際のサイズを設定するメソッド。通常、クライアントが本メソッドを呼ぶ必要はない。
  void SetActualSize(u64 size) { actualSize = size; }

  /// 格納アイテムの格納データサイズを取得するメソッド。
  u64  GetStoredSize() const { return storedSize; }
  /// 格納アイテムの格納データサイズを設定するメソッド。通常、クライアントが本メソッドを呼ぶ必要はない。
  void SetStoredSize(u64 size) { storedSize = size; }

  /// 格納アイテムの属性を取得するメソッド。
  u32  GetAttributes() const { return fileAttributes.GetAttributes(); }
  /// 格納アイテムの属性を設定するメソッド。
  void SetAttributes(u32 attr) { fileAttributes.SetAttributes(attr); }
  /// 格納アイテムのファイル属性を取得するメソッド。
  FileAttributes GetFileAttributes() const { return fileAttributes; }

  /// 格納アイテムの作成日時を取得するメソッド。64 整数値が 0 のときは、適切な値が設定されていないことを意味する。
  Time GetCreated() const { return created; }
  /// 格納アイテムの作成日時を設定するメソッド。
  void SetCreated(Time time) { created = time; }

  /// 格納アイテムの最終アクセス日時を取得するメソッド。64 整数値が 0 のときは、適切な値が設定されていないことを意味する。
  Time GetAccessed() const { return accessed; }
  /// 格納アイテムの最終アクセス日時を設定するメソッド。
  void SetAccessed(Time time) { accessed = time; }

  /// 格納アイテムの最終更新日時を取得するメソッド。64 整数値が 0 のときは、適切な値が設定されていないことを意味する。
  Time GetModified() const { return modified; }
  /// 格納アイテムの最終更新日時を設定するメソッド。
  void SetModified(Time time) { modified = time; }

  /// ArchiveContent などが管理している格納アイテムリストにおけるインデックスを取得するメソッド。
  index_t GetItemIndex() const { return itemIndex; }
  /// ArchiveContent などが管理している格納アイテムリストにおけるインデックスを設定するメソッド。
  void SetItemIndex(index_t index) { itemIndex = index; }

  /// 親コンテナのポインタを取得するメソッド。NULL が返される場合は、まだアーカイブの仮想階層構造が構築されていないことを表す。
  StoredItemContainer *GetParent() { return parent; }
  /// 親コンテナのポインタを設定するメソッド。
  void SetParent(StoredItemContainer *p) { parent = p; }

private:

  FileAttributes  fileAttributes; /// ファイル属性。
//u32             fields;         /// ソリッドとか分割とかのフラグを含むビットフィールド。
//u32             method;         /// 圧縮メソッド。
//u32             crc;            /// CRC

  Time            modified;   /// 最終変更日時。
  Time            created;    /// 作成日時。
  Time            accessed;   /// 最終アクセス日時。

  u64             actualSize; /// 展開時のサイズ。
  u64             storedSize; /// 圧縮時のサイズ。

  /// <summary>
  /// 低水準アーカイバ（7-Zip コーデックとか）が保持している格納ファイルリストにおけるインデックス。
  /// </summary>
  /// <remarks>
  /// このメンバの値が INVALID_T であるときは、対応する要素が格納ファイルリストに存在しないことを意味する。
  /// たとえば、通常のアーカイブではフォルダ要素はアーカイブ内に実体を持たないが、そのようなとき本メンバは
  /// INVALID_T になっている。
  /// <remarks>
  index_t itemIndex;

  /// 親コンテナのポインタ（所有権なし）。
  StoredItemContainer *parent;

  /// <summary>
  /// 正規化済みの相対パス文字列（要するにファイル名／ディレクトリ名）。
  /// </summary>
  /// <remarks>
  /// ディレクトリアイテムであったとしても末尾にバックスラッシュは<b>付けない</b>。
  /// ディレクトリかどうかはあくまでファイル属性で判断する。<p>
  /// 展開時に必要となるルートからの相対パスは、上位コンテナを辿って動的に作成する。
  /// なぜなら、tar 対応で上位フォルダの名称が自動的に変更されていることがあるので、
  /// 格納アイテムの相対パス文字列をそのまま利用して展開するとは限らないからである。
  /// <remarks>
  szstring name;

};

SZ_NS_END(szpp)

#endif // !defined(INCLUDED_SZ_STORED_ITEM)
